<?php
/*--------------------------------------------------------------
   Mapper.php 2021-08-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Data\Data;

use Gambio\Admin\Modules\Statistics\App\Data\Factory;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Serial as SerialData;

/**
 * Class Mapper
 *
 * @package Gambio\Admin\Modules\Statistics\App\Data\Data
 */
class Mapper
{
    /**
     * @var Factory
     */
    private $factory;
    
    
    /**
     * Constructor.
     */
    public function __construct(Factory $factory)
    {
        $this->factory = $factory;
    }
    
    
    /**
     * Return orders.
     */
    public function mapOrders(array $orders): SerialData
    {
        $serialFactory = $this->factory->useSerialData();
        
        $categories = [];
        $values     = [];
        
        foreach ($orders as ['orders_date' => $category, 'orders_count' => $value]) {
            $categories[] = $serialFactory->createCategory($category);
            $values[]     = $serialFactory->createItemValue((float)$value);
        }
        
        return $serialFactory->createSerialData($serialFactory->createCategories(...$categories),
                                                $serialFactory->createItems($serialFactory->createItem($serialFactory->createItemTitle("orders"),
                                                                                                       $serialFactory->createItemValues(...
                                                                                                           $values))));
    }
    
    
    /**
     * Return sales.
     */
    public function mapSales(array $sales): SerialData
    {
        $serialFactory = $this->factory->useSerialData();
        
        $categories = [];
        $values     = [];
        $taxes      = [];
        
        foreach ($sales['taxes'] as $tax) {
            $taxes[$tax['purchased_date']] = $tax['taxes'];
        }
        
        if (count($sales['orders'])) {
            foreach ($sales['orders'] as ['purchased_date' => $category, 'sales' => $sales]) {
                $categories[] = $serialFactory->createCategory($category);
                $values[]     = $serialFactory->createItemValue(isset($taxes[$category]) ? (float)$sales
                                                                                           - (float)$taxes[$category] : (float)$sales);
            }
        }
        
        return $serialFactory->createSerialData($serialFactory->createCategories(...$categories),
                                                $serialFactory->createItems($serialFactory->createItem($serialFactory->createItemTitle("sales"),
                                                                                                       $serialFactory->createItemValues(...
                                                                                                           $values))));
    }
    
    
    /**
     * Return visitors.
     */
    public function mapVisitors(array $visitors): SerialData
    {
        $serialFactory = $this->factory->useSerialData();
        
        $categories = [];
        $values     = [];
        
        foreach ($visitors as ['date' => $category, 'amount' => $value]) {
            $categories[] = $serialFactory->createCategory($category);
            $values[]     = $serialFactory->createItemValue((float)$value);
        }
        
        return $serialFactory->createSerialData($serialFactory->createCategories(...$categories),
                                                $serialFactory->createItems($serialFactory->createItem($serialFactory->createItemTitle("visitors"),
                                                                                                       $serialFactory->createItemValues(...
                                                                                                           $values))));
    }
    
    
    /**
     * Return conversion rates.
     */
    public function mapConversionRates(array $conversionRates): SerialData
    {
        $serialFactory = $this->factory->useSerialData();
        
        $categories = [];
        $values     = [];
        
        foreach ($conversionRates['orders'] as ['purchased_date' => $purchaseDate, 'purchases' => $purchases]) {
            foreach ($conversionRates['visitors'] as ['visitor_date' => $visitorDate, 'visitors' => $visitors]) {
                if ($purchaseDate !== $visitorDate) {
                    continue;
                }
                
                $categories[] = $serialFactory->createCategory($purchaseDate);
                $values[]     = $serialFactory->createItemValue(round((float)$purchases / (float)$visitors * 100,
                                                                      2));
            }
        }
        
        return $serialFactory->createSerialData($serialFactory->createCategories(...$categories),
                                                $serialFactory->createItems($serialFactory->createItem($serialFactory->createItemTitle("conversion"),
                                                                                                       $serialFactory->createItemValues(...
                                                                                                           $values))));
    }
}